
//
// Copyright (C) 2025 voidtools / David Carpenter
// 
// Permission is hereby granted, free of charge, 
// to any person obtaining a copy of this software 
// and associated documentation files (the "Software"), 
// to deal in the Software without restriction, 
// including without limitation the rights to use, 
// copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit 
// persons to whom the Software is furnished to do so, 
// subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be 
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
// OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
// TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
// SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// property IDs and information on properties.
// known ES conflicts:
// -version, use -add-column version
// -path, use -pathpart or -pathcolumn

#include "es.h"

static int _property_name_compare(const ES_UTF8 *name,const wchar_t *search);

// property name to IDs are sorted alphabetically.
// the '-' in the name is ignored when sorting.
// name is always lowercase ASCII.
#define PROPERTY_NAME_TO_ID_MACRO(name,property_id)	{name,property_id},

static const property_name_to_id_t _property_name_to_id_array[] = 
{
	#include "property_name_to_id_macro.h"
};

#undef PROPERTY_NAME_TO_ID_MACRO

#define PROPERTY_NAME_TO_ID_COUNT	(sizeof(_property_name_to_id_array) / sizeof(const property_name_to_id_t))

// property canonical names
// sorted by ID.
static const ES_UTF8 *_property_name_array[EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT] = 
{
	"Name",
	"Path",
	"Size",
	"Extension",
	"Type",
	"Date Modified",
	"Date Created",
	"Date Accessed",
	"Attributes",
	"Date Recently Changed",
	"Run Count",
	"Date Run",
	"File List Name",
	"Width",
	"Height",
	"Dimensions",
	"Aspect Ratio",
	"Bit Depth",
	"Length",
	"Audio Sample Rate",
	"Audio Channels",
	"Audio Bits Per Sample",
	"Audio Bit Rate",
	"Audio Format",
	"File Signature",
	"Title",
	"Artist",
	"Album",
	"Year",
	"Comment",
	"Track",
	"Genre",
	"Frame Rate",
	"Video Bit Rate",
	"Video Format",
	"Rating",
	"Tags",
	"MD5",
	"SHA-1",
	"SHA-256",
	"CRC-32",
	"Size on Disk",
	"Description",
	"Version",
	"Product Name",
	"Product Version",
	"Company",
	"Kind",
	"Name Length",
	"Path and Name Length",
	"Subject",
	"Authors",
	"Date Taken",
	"Software",
	"Date Acquired",
	"Copyright",
	"Image ID",
	"Horizontal Resolution",
	"Vertical Resolution",
	"Compression",
	"Resolution Unit",
	"Color Representation",
	"Compressed Bits Per Pixel",
	"Camera Maker",
	"Camera Model",
	"F-Stop",
	"Exposure Time",
	"ISO Speed",
	"Exposure Bias",
	"Focal Length",
	"Max Aperture",
	"Metering Mode",
	"Subject Distance",
	"Flash Mode",
	"Flash Energy",
	"35mm Focal Length",
	"Lens Maker",
	"Lens Model",
	"Flash Maker",
	"Flash Model",
	"Camera Serial Number",
	"Contrast",
	"Brightness",
	"Light Source",
	"Exposure Program",
	"Saturation",
	"Sharpness",
	"White Balance",
	"Photometric Interpretation",
	"Digital Zoom",
	"Exif Version",
	"Latitude",
	"Longitude",
	"Altitude",
	"Subtitle",
	"Total Bit Rate",
	"Director",
	"Producer",
	"Writer",
	"Publisher",
	"Content Provider",
	"Media Created",
	"Encoded By",
	"Author URL",
	"Promotion URL",
	"Offline Availability",
	"Offline Status",
	"Shared With",
	"Owner",
	"Computer",
	"Album Artist",
	"Parental Rating Reason",
	"Composer",
	"Conductor",
	"Group Description",
	"Mood",
	"Part of Set",
	"Initial Key",
	"Beats Per Minute",
	"Protected",
	"Part of a Compilation",
	"Parental Rating",
	"Period",
	"People",
	"Category",
	"Content Status",
	"Document Content Type",
	"Page Count",
	"Word Count",
	"Character Count",
	"Line Count",
	"Paragraph Count",
	"Template",
	"Scale",
	"Links Dirty",
	"Language",
	"Last Author",
	"Revision Number",
	"Version Number",
	"Manager",
	"Content Created",
	"Date Saved",
	"Date Printed",
	"Total Editing Time",
	"Original Filename",
	"Date Released",
	"Slide Count",
	"Note Count",
	"Hidden Slide Count",
	"Presentation Format",
	"Trademarks",
	"Display Name",
	"Name UTF-8 Length in UTF-8 Bytes",
	"Path and Name Length in UTF-8 Bytes",
	"Child Count",
	"Child Folder Count",
	"Child File Count",
	"Child Count From Disk",
	"Child Folder Count From Disk",
	"Child File Count From Disk",
	"Folder Depth",
	"Total Size",
	"Total Size On Disk",
	"Date Changed",
	"Hard Link Count",
	"Delete Pending",
	"Is Directory",
	"Alternate Data Stream Count",
	"Alternate Data Stream Names",
	"Total Alternate Data Stream Size",
	"Total Alternate Data Stream Size On Disk",
	"Compressed Size",
	"Compression Format",
	"Compression Unit Shift",
	"Compression Chunk Shift",
	"Compression Cluster Shift",
	"Compression Ratio",
	"Reparse Tag",
	"Remote Protocol",
	"Remote Protocol Version",
	"Remote Protocol Flags",
	"Logical Bytes Per Sector",
	"Physical Bytes Per Sector For Atomicity",
	"Physical Bytes Per Sector For Performance",
	"Effective Physical Bytes Per Sector For Atomicity",
	"File Storage Info Flags",
	"Byte Offset For Sector Alignment",
	"Byte Offset For Partition Alignment",
	"Alignment Requirement",
	"Volume Serial Number",
	"File ID",
	"Frame Count",
	"Cluster Size",
	"Sector Size",
	"Available Free Disk Size",
	"Free Disk Size",
	"Total Disk Size",
	"UNUSED Volume Label",
	"Maximum Component Length",
	"File System Flags",
	"File System",
	"Orientation",
	"End Of File",
	"Short Name",
	"Short Path and Name",
	"Encryption Status",
	"Hard Link Filenames",
	"Index Type",
	"Drive Type",
	"Binary Type",
	"Regular Expression Match 0",
	"Regular Expression Match 1",
	"Regular Expression Match 2",
	"Regular Expression Match 3",
	"Regular Expression Match 4",
	"Regular Expression Match 5",
	"Regular Expression Match 6",
	"Regular Expression Match 7",
	"Regular Expression Match 8",
	"Regular Expression Match 9",
	"Sibling Count",
	"Sibling Folder Count",
	"Sibling File Count",
	"Index Number",
	"Shortcut Target",
	"Out of Date",
	"Incur Seek Penalty",
	"Plain Text Line Count",
	"Aperture",
	"Maker Note",
	"Related Sound File",
	"Shutter Speed",
	"Transcoded For Sync",
	"Case Sensitive Dir",
	"Date Indexed",
	"Name Frequency",
	"Size Frequency",
	"Extension Frequency",
	"Regular Expression Matches 1-9",
	"URL",
	"Filename", // Path and Name
	"Parent File ID",
	"SHA-512",
	"SHA-384",
	"CRC-64",
	"First Byte",
	"First 2 Bytes",
	"First 4 Bytes",
	"First 8 Bytes",
	"First 16 Bytes",
	"First 32 Bytes",
	"First 64 Bytes",
	"First 128 Bytes",
	"Last Byte",
	"Last 2 Bytes",
	"Last 4 Bytes",
	"Last 8 Bytes",
	"Last 16 Bytes",
	"Last 32 Bytes",
	"Last 64 Bytes",
	"Last 128 Bytes",
	"Byte Order Mark",
	"Volume Label",
	"File List Path and Name",
	"Display Path and Name",
	"Parse Name",
	"Parse Path and Name",
	"Stem",
	"Shell Attributes",
	"Is Folder",
	"Valid UTF-8",
	"Stem Length",
	"Extension Length",
	"Path Length",
	"Date Modified Time",
	"Date Created Time",
	"Date Accessed Time",
	"Date Modified Date",
	"Date Created Date",
	"Date Accessed Date",
	"Parent Name",
	"Reparse Target",
	"Descendant Count",
	"Descendant Folder Count",
	"Descendant File Count",
	"From",
	"To",
	"Date Received",
	"Date Sent",
	"Container Filenames",
	"Container File Count",
	"Custom Property 0",
	"Custom Property 1",
	"Custom Property 2",
	"Custom Property 3",
	"Custom Property 4",
	"Custom Property 5",
	"Custom Property 6",
	"Custom Property 7",
	"Custom Property 8",
	"Custom Property 9",
	"Allocation Size",
	"SFV CRC-32",
	"md5sum MD5",
	"sha1sum SHA-1",
	"sha256sum SHA-256",
	"SFV Pass",
	"md5sum Pass",
	"sha1sum Pass",
	"sha256sum Pass",
	"Alternate Data Stream ANSI",
	"Alternate Data Stream UTF-8",
	"Alternate Data Stream UTF-16",
	"Alternate Data Stream UTF-16BE",
	"Alternate Data Stream Text/Plain",
	"Alternate Data Stream Hex",
	"Perceived Type",
	"Content Type",
	"Opened By",
	"Target Machine",
	"sha512sum SHA-512",
	"sha512sum Pass",
	"Parent Path",
	"First 256 Bytes",
	"First 512 Bytes",
	"Last 256 Bytes",
	"Last 512 Bytes",
	"Online",
	"Column 0",
	"Column 1",
	"Column 2",
	"Column 3",
	"Column 4",
	"Column 5",
	"Column 6",
	"Column 7",
	"Column 8",
	"Column 9",
	"Column A",
	"Column B",
	"Column C",
	"Column D",
	"Column E",
	"Column F",
	"Zone ID",
	"Referrer URL",
	"Host URL",
	"Character Encoding",
	"Root Name",
	"Used Disk Size",
	"Volume Path",
	"Max Child Depth",
	"Total Child Size",
	"Row",
	"Child Occurrence Count",
	"Volume Name",
	"Descendant Occurrence Count",
	"Object ID",
	"Birth Volume ID",
	"Birth Object ID",
	"Domain ID",
	"Folder Data CRC-32",
	"Folder Data CRC-64",
	"Folder Data MD5",
	"Folder Data SHA-1",
	"Folder Data SHA-256",
	"Folder Data SHA-512",
	"Folder Data and Names CRC-32",
	"Folder Data and Names CRC-64",
	"Folder Data and Names MD5",
	"Folder Data and Names SHA-1",
	"Folder Data and Names SHA-256",
	"Folder Data and Names SHA-512",
	"Folder Names CRC-32",
	"Folder Names CRC-64",
	"Folder Names MD5",
	"Folder Names SHA-1",
	"Folder Names SHA-256",
	"Folder Names SHA-512",
	"Folder Data CRC-32 From Disk",
	"Folder Data CRC-64 From Disk",
	"Folder Data MD5 From Disk",
	"Folder Data SHA-1 From Disk",
	"Folder Data SHA-256 From Disk",
	"Folder Data SHA-512 From Disk",
	"Folder Data and Names CRC-32 From Disk",
	"Folder Data and Names CRC-64 From Disk",
	"Folder Data and Names MD5 From Disk",
	"Folder Data and Names SHA-1 From Disk",
	"Folder Data and Names SHA-256 From Disk",
	"Folder Data and Names SHA-512 From Disk",
	"Folder Names CRC-32 From Disk",
	"Folder Names CRC-64 From Disk",
	"Folder Names MD5 From Disk",
	"Folder Names SHA-1 From Disk",
	"Folder Names SHA-256 From Disk",
	"Folder Names SHA-512 From Disk",
	"Long Name",
	"Long Path and Name",
	"Digital Signature Name",
	"Digital Signature Timestamp",
	"Audio Track Count",
	"Video Track Count",
	"Subtitle Track Count",
	"Network Index Host",
	"Original Location",
	"Date Deleted",
	"Status",
	"Vorbis Comment",
	"QuickTime Metadata",
	"Parent Size",
	"Root Size",
	"Opens With",
	"Randomize",
	"Icon",
	"Thumbnail",
	"Content",
	"-",
};

// IPC2 sort properties (ES_COLUMN_FILENAME => EVERYTHING3_PROPERTY_ID_PATH_AND_NAME)
#define PROPERTY_OLD_COLUMN_MACRO(property_id)	property_id,

static const DWORD _property_old_column_id_to_property_id_array[] = 
{
	#include "property_old_column_macro.h"
};

#undef PROPERTY_OLD_COLUMN_MACRO

#define PROPERTY_OLD_COLUMN_COUNT	(sizeof(_property_old_column_id_to_property_id_array) / sizeof(DWORD))

// format type of each property
// from the format we know how to display the data, work out the left/right alignment and the default column width.

const BYTE es_property_format[EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT] = 
{
	PROPERTY_FORMAT_TEXT30, 				// EVERYTHING3_PROPERTY_ID_NAME,
	PROPERTY_FORMAT_TEXT47, 				// EVERYTHING3_PROPERTY_ID_PATH,
	PROPERTY_FORMAT_SIZE, 					// EVERYTHING3_PROPERTY_ID_SIZE,
	PROPERTY_FORMAT_EXTENSION, 				// EVERYTHING3_PROPERTY_ID_EXTENSION,
	PROPERTY_FORMAT_TEXT30,			 		// EVERYTHING3_PROPERTY_ID_TYPE,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_MODIFIED,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_CREATED,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_ACCESSED,
	PROPERTY_FORMAT_ATTRIBUTES, 			// EVERYTHING3_PROPERTY_ID_ATTRIBUTES,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_RECENTLY_CHANGED,
	PROPERTY_FORMAT_GROUPING_NUMBER6, 		// EVERYTHING3_PROPERTY_ID_RUN_COUNT,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_RUN,
	PROPERTY_FORMAT_TEXT30, 				// EVERYTHING3_PROPERTY_ID_FILE_LIST_NAME,
	PROPERTY_FORMAT_NOGROUPING_NUMBER4, 	// EVERYTHING3_PROPERTY_ID_WIDTH,
	PROPERTY_FORMAT_NOGROUPING_NUMBER4, 	// EVERYTHING3_PROPERTY_ID_HEIGHT,
	PROPERTY_FORMAT_DIMENSIONS,				// EVERYTHING3_PROPERTY_ID_DIMENSIONS,
	PROPERTY_FORMAT_ASPECT_RATIO,			// EVERYTHING3_PROPERTY_ID_ASPECT_RATIO,
	PROPERTY_FORMAT_NOGROUPING_NUMBER2, 	// EVERYTHING3_PROPERTY_ID_BIT_DEPTH,
	PROPERTY_FORMAT_DURATION, 				// EVERYTHING3_PROPERTY_ID_LENGTH,
	PROPERTY_FORMAT_KHZ, 					// EVERYTHING3_PROPERTY_ID_AUDIO_SAMPLE_RATE,
	PROPERTY_FORMAT_NOGROUPING_NUMBER1, 	// EVERYTHING3_PROPERTY_ID_AUDIO_CHANNELS,
	PROPERTY_FORMAT_NOGROUPING_NUMBER2, 	// EVERYTHING3_PROPERTY_ID_AUDIO_BITS_PER_SAMPLE,
	PROPERTY_FORMAT_KBPS,	 				// EVERYTHING3_PROPERTY_ID_AUDIO_BIT_RATE,
	PROPERTY_FORMAT_TEXT16, 				// EVERYTHING3_PROPERTY_ID_AUDIO_FORMAT,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_FILE_SIGNATURE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_TITLE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_ARTIST,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_ALBUM,
	PROPERTY_FORMAT_NOGROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_YEAR,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COMMENT,
	PROPERTY_FORMAT_NOGROUPING_NUMBER3,		// EVERYTHING3_PROPERTY_ID_TRACK,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_GENRE,	
	PROPERTY_FORMAT_FIXED_Q1K,				// EVERYTHING3_PROPERTY_ID_FRAME_RATE,
	PROPERTY_FORMAT_KBPS, 					// EVERYTHING3_PROPERTY_ID_VIDEO_BIT_RATE,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_VIDEO_FORMAT,
	PROPERTY_FORMAT_RATING, 				// EVERYTHING3_PROPERTY_ID_RATING,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_TAGS,
	PROPERTY_FORMAT_DATA16, 				// EVERYTHING3_PROPERTY_ID_MD5,
	PROPERTY_FORMAT_DATA20, 				// EVERYTHING3_PROPERTY_ID_SHA1,
	PROPERTY_FORMAT_DATA32, 				// EVERYTHING3_PROPERTY_ID_SHA256,
	PROPERTY_FORMAT_DATA4, 					// EVERYTHING3_PROPERTY_ID_CRC32,
	PROPERTY_FORMAT_SIZE, 					// EVERYTHING3_PROPERTY_ID_SIZE_ON_DISK,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_DESCRIPTION,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_VERSION,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_PRODUCT_NAME,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_PRODUCT_VERSION,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_COMPANY,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_KIND,
	PROPERTY_FORMAT_NOGROUPING_NUMBER3, 	// EVERYTHING3_PROPERTY_ID_NAME_LENGTH,
	PROPERTY_FORMAT_NOGROUPING_NUMBER3, 	// EVERYTHING3_PROPERTY_ID_PATH_AND_NAME_LENGTH,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_SUBJECT,
	PROPERTY_FORMAT_TEXT32,			 		// EVERYTHING3_PROPERTY_ID_AUTHORS,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_TAKEN,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_SOFTWARE,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_ACQUIRED,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_COPYRIGHT,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_IMAGE_ID,
	PROPERTY_FORMAT_NOGROUPING_NUMBER4, 	// EVERYTHING3_PROPERTY_ID_HORIZONTAL_RESOLUTION,
	PROPERTY_FORMAT_NOGROUPING_NUMBER4, 	// EVERYTHING3_PROPERTY_ID_VERTICAL_RESOLUTION,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_COMPRESSION,
	PROPERTY_FORMAT_FORMATTED_TEXT32,	 		// EVERYTHING3_PROPERTY_ID_RESOLUTION_UNIT,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_COLOR_REPRESENTATION,
	PROPERTY_FORMAT_FIXED_Q1K,				// EVERYTHING3_PROPERTY_ID_COMPRESSED_BITS_PER_PIXEL,	
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_CAMERA_MAKER,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_CAMERA_MODEL,
	PROPERTY_FORMAT_F_STOP,					// EVERYTHING3_PROPERTY_ID_F_STOP,
	PROPERTY_FORMAT_EXPOSURE_TIME,			// EVERYTHING3_PROPERTY_ID_EXPOSURE_TIME,
	PROPERTY_FORMAT_ISO_SPEED, 				// EVERYTHING3_PROPERTY_ID_ISO_SPEED,
	PROPERTY_FORMAT_EXPOSURE_BIAS, 			// EVERYTHING3_PROPERTY_ID_EXPOSURE_BIAS,
	PROPERTY_FORMAT_FOCAL_LENGTH, 			// EVERYTHING3_PROPERTY_ID_FOCAL_LENGTH,
	PROPERTY_FORMAT_FIXED_Q1M,				// EVERYTHING3_PROPERTY_ID_MAX_APERTURE,
	PROPERTY_FORMAT_FORMATTED_TEXT24, 			// EVERYTHING3_PROPERTY_ID_METERING_MODE,
	PROPERTY_FORMAT_SUBJECT_DISTANCE,		// EVERYTHING3_PROPERTY_ID_SUBJECT_DISTANCE,
	PROPERTY_FORMAT_FORMATTED_TEXT32, 			// EVERYTHING3_PROPERTY_ID_FLASH_MODE,
	PROPERTY_FORMAT_BCPS,					// EVERYTHING3_PROPERTY_ID_FLASH_ENERGY,
	PROPERTY_FORMAT_35MM_FOCAL_LENGTH, 		// EVERYTHING3_PROPERTY_ID_35MM_FOCAL_LENGTH,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_LENS_MAKER,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_LENS_MODEL,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_FLASH_MAKER,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_FLASH_MODEL,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_CAMERA_SERIAL_NUMBER,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_CONTRAST,
	PROPERTY_FORMAT_FIXED_Q1M,				// EVERYTHING3_PROPERTY_ID_BRIGHTNESS,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_LIGHT_SOURCE,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_EXPOSURE_PROGRAM,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_SATURATION,
	PROPERTY_FORMAT_FORMATTED_TEXT16, 			// EVERYTHING3_PROPERTY_ID_SHARPNESS,
	PROPERTY_FORMAT_FORMATTED_TEXT8, 			// EVERYTHING3_PROPERTY_ID_WHITE_BALANCE,
	PROPERTY_FORMAT_FORMATTED_TEXT8, 			// EVERYTHING3_PROPERTY_ID_PHOTOMETRIC_INTERPRETATION,
	PROPERTY_FORMAT_FIXED_Q1K,				// EVERYTHING3_PROPERTY_ID_DIGITAL_ZOOM,
	PROPERTY_FORMAT_TEXT8, 					// EVERYTHING3_PROPERTY_ID_EXIF_VERSION,
	PROPERTY_FORMAT_FIXED_Q1M, 				// EVERYTHING3_PROPERTY_ID_LATITUDE,
	PROPERTY_FORMAT_FIXED_Q1M, 				// EVERYTHING3_PROPERTY_ID_LONGITUDE,
	PROPERTY_FORMAT_ALTITUDE, 				// EVERYTHING3_PROPERTY_ID_ALTITUDE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_SUBTITLE
	PROPERTY_FORMAT_KBPS,					// EVERYTHING3_PROPERTY_ID_TOTAL_BIT_RATE
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_DIRECTORS,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PRODUCERS,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_WRITERS,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PUBLISHER,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CONTENT_DISTRIBUTOR,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_ENCODED,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_ENCODED_BY,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_AUTHOR_URL,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_PROMOTION_URL,
	PROPERTY_FORMAT_FORMATTED_TEXT32,			// EVERYTHING3_PROPERTY_ID_OFFLINE_AVAILABILITY,
	PROPERTY_FORMAT_FORMATTED_TEXT32,			// EVERYTHING3_PROPERTY_ID_OFFLINE_STATUS,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_SHARED_WITH,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_OWNER,
	PROPERTY_FORMAT_TEXT16,					// EVERYTHING3_PROPERTY_ID_COMPUTER,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_ALBUM_ARTIST,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PARENTAL_RATING_REASON,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COMPOSER,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CONDUCTOR,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CONTENT_GROUP_DESCRIPTION,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_MOOD,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_PART_OF_SET,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_INITIAL_KEY,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_BEATS_PER_MINUTE,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_PROTECTED,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_PART_OF_A_COMPILATION,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_PARENTAL_RATING,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PERIOD,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PEOPLE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CATEGORY,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CONTENT_STATUS,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_DOCUMENT_CONTENT_TYPE,
	PROPERTY_FORMAT_GROUPING_NUMBER3,		// EVERYTHING3_PROPERTY_ID_PAGE_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_WORD_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER6,		// EVERYTHING3_PROPERTY_ID_CHARACTER_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_LINE_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_PARAGRAPH_COUNT,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_TEMPLATE,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_SCALE,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_LINKS_DIRTY,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_LANGUAGE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_LAST_AUTHOR,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_REVISION_NUMBER,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_VERSION_NUMBER,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_MANAGER,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_CONTENT_CREATED,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_SAVED,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_PRINTED,
	PROPERTY_FORMAT_DURATION,				// EVERYTHING3_PROPERTY_ID_TOTAL_EDITING_TIME,	
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_ORIGINAL_FILE_NAME,
	PROPERTY_FORMAT_TEXT10,					// EVERYTHING3_PROPERTY_ID_DATE_RELEASED,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_SLIDE_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER3,		// EVERYTHING3_PROPERTY_ID_NOTE_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_HIDDEN_SLIDE_COUNT,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PRESENTATION_FORMAT,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_TRADEMARKS,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_DISPLAY_NAME,
	PROPERTY_FORMAT_NOGROUPING_NUMBER3,		// EVERYTHING3_PROPERTY_ID_NAME_LENGTH_IN_UTF8_BYTES
	PROPERTY_FORMAT_NOGROUPING_NUMBER3,		// EVERYTHING3_PROPERTY_ID_PATH_AND_NAME_LENGTH_IN_UTF8_BYTES
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CHILD_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CHILD_FOLDER_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CHILD_FILE_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CHILD_COUNT_FROM_DISK,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CHILD_FOLDER_COUNT_FROM_DISK,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CHILD_FILE_COUNT_FROM_DISK,
	PROPERTY_FORMAT_GROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_FOLDER_DEPTH,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_TOTAL_SIZE,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_TOTAL_SIZE_ON_DISK,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_CHANGED,
	PROPERTY_FORMAT_GROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_HARD_LINK_COUNT,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_DELETE_PENDING,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_IS_DIRECTORY,
	PROPERTY_FORMAT_GROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_COUNT,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_NAMES,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_TOTAL_ALTERNATE_DATA_STREAM_SIZE,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_TOTAL_ALTERNATE_DATA_STREAM_SIZE_ON_DISK,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_COMPRESSED_SIZE,
	PROPERTY_FORMAT_FORMATTED_TEXT16,			// EVERYTHING3_PROPERTY_ID_COMPRESSION_FORMAT,
	PROPERTY_FORMAT_GROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_COMPRESSION_UNIT_SHIFT,
	PROPERTY_FORMAT_GROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_COMPRESSION_CHUNK_SHIFT,
	PROPERTY_FORMAT_GROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_COMPRESSION_CLUSTER_SHIFT,
	PROPERTY_FORMAT_PERCENT,				// EVERYTHING3_PROPERTY_ID_COMPRESSION_RATIO
	PROPERTY_FORMAT_HEX_NUMBER8,			// EVERYTHING3_PROPERTY_ID_REPARSE_TAG
	PROPERTY_FORMAT_HEX_NUMBER8,			// EVERYTHING3_PROPERTY_ID_REMOTE_PROTOCOL,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_REMOTE_PROTOCOL_VERSION
	PROPERTY_FORMAT_HEX_NUMBER8,			// EVERYTHING3_PROPERTY_ID_REMOTE_PROTOCOL_FLAGS,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_LOGICAL_BYTES_PER_SECTOR,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_PHYSICAL_BYTES_PER_SECTOR_FOR_ATOMICITY,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_PHYSICAL_BYTES_PER_SECTOR_FOR_PERFORMANCE,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_EFFECTIVE_PHYSICAL_BYTES_PER_SECTOR_FOR_ATOMICITY,
	PROPERTY_FORMAT_HEX_NUMBER8,			// EVERYTHING3_PROPERTY_ID_FILE_STORAGE_INFO_FLAGS,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_BYTE_OFFSET_FOR_SECTOR_ALIGNMENT,
	PROPERTY_FORMAT_GROUPING_NUMBER7,		// EVERYTHING3_PROPERTY_ID_BYTE_OFFSET_FOR_PARTITION_ALIGNMENT,
	PROPERTY_FORMAT_GROUPING_NUMBER7,		// EVERYTHING3_PROPERTY_ID_ALIGNMENT_REQUIREMENT,
	PROPERTY_FORMAT_FORMATTED_TEXT16,			// EVERYTHING3_PROPERTY_ID_VOLUME_SERIAL_NUMBER,
	PROPERTY_FORMAT_HEX_NUMBER32,			// EVERYTHING3_PROPERTY_ID_FILE_ID,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_FRAME_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CLUSTER_SIZE,
	PROPERTY_FORMAT_GROUPING_NUMBER4,		// EVERYTHING3_PROPERTY_ID_SECTOR_SIZE,
	PROPERTY_FORMAT_VOLUME_SIZE,			// EVERYTHING3_PROPERTY_ID_AVAILABLE_FREE_DISK_SIZE, 
	PROPERTY_FORMAT_VOLUME_SIZE,			// EVERYTHING3_PROPERTY_ID_FREE_DISK_SIZE,
	PROPERTY_FORMAT_VOLUME_SIZE,			// EVERYTHING3_PROPERTY_ID_TOTAL_DISK_SIZE,
	PROPERTY_FORMAT_TEXT32,					// __EVERYTHING3_PROPERTY_ID_VOLUME_LABEL,
	PROPERTY_FORMAT_NOGROUPING_NUMBER3,		// EVERYTHING3_PROPERTY_ID_MAXIMUM_COMPONENT_LENGTH,
	PROPERTY_FORMAT_HEX_NUMBER8,			// EVERYTHING3_PROPERTY_ID_FILE_SYSTEM_FLAGS,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_FILE_SYSTEM,
	PROPERTY_FORMAT_FORMATTED_TEXT24,			// EVERYTHING3_PROPERTY_ID_ORIENTATION,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_END_OF_FILE,
	PROPERTY_FORMAT_TEXT12,					// EVERYTHING3_PROPERTY_ID_SHORT_NAME, 
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_SHORT_PATH_AND_NAME,
	PROPERTY_FORMAT_FORMATTED_TEXT24,			// EVERYTHING3_PROPERTY_ID_ENCRYPTION_STATUS,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_HARD_LINK_FILE_NAMES,
	PROPERTY_FORMAT_FORMATTED_TEXT24,			// EVERYTHING3_PROPERTY_ID_INDEX_TYPE,	
	PROPERTY_FORMAT_FORMATTED_TEXT12,			// EVERYTHING3_PROPERTY_ID_DRIVE_TYPE,
	PROPERTY_FORMAT_FORMATTED_TEXT16,			// EVERYTHING3_PROPERTY_ID_BINARY_TYPE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_0,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_1,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_2,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_3,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_4,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_5,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_6,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_7,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_8,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCH_9,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_SIBLING_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_SIBLING_FOLDER_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_SIBLING_FILE_COUNT,
	PROPERTY_FORMAT_NOGROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_INDEX_NUMBER,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_SHORTCUT_TARGET
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_OUT_OF_DATE,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_INCUR_SEEK_PENALTY,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_PLAIN_TEXT_LINE_COUNT,
	PROPERTY_FORMAT_FIXED_Q1M,				// EVERYTHING3_PROPERTY_ID_APERTURE,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_MAKER_NOTE,
	PROPERTY_FORMAT_TEXT32, 				// EVERYTHING3_PROPERTY_ID_RELATED_SOUND_FILE,
	PROPERTY_FORMAT_SEC,					// EVERYTHING3_PROPERTY_ID_SHUTTER_SPEED,
	PROPERTY_FORMAT_YESNO, 					// EVERYTHING3_PROPERTY_ID_TRANSCODED_FOR_SYNC,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_CASE_SENSITIVE_DIR,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_INDEXED,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_NAME_FREQUENCY,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_SIZE_FREQUENCY,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_EXTENSION_FREQUENCY,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_REGEX_MATCHES,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_URL,
	PROPERTY_FORMAT_TEXT47,					// EVERYTHING3_PROPERTY_ID_PATH_AND_NAME,
	PROPERTY_FORMAT_HEX_NUMBER32,			// EVERYTHING3_PROPERTY_ID_PARENT_FILE_ID,
	PROPERTY_FORMAT_DATA64, 				// EVERYTHING3_PROPERTY_ID_SHA512,
	PROPERTY_FORMAT_DATA48, 				// EVERYTHING3_PROPERTY_ID_SHA384,
	PROPERTY_FORMAT_DATA8, 					// EVERYTHING3_PROPERTY_ID_CRC64,
	PROPERTY_FORMAT_DATA1, 					// EVERYTHING3_PROPERTY_ID_FIRST_BYTE,
	PROPERTY_FORMAT_DATA2, 					// EVERYTHING3_PROPERTY_ID_FIRST_2_BYTES,
	PROPERTY_FORMAT_DATA4, 					// EVERYTHING3_PROPERTY_ID_FIRST_4_BYTES,
	PROPERTY_FORMAT_DATA8, 					// EVERYTHING3_PROPERTY_ID_FIRST_8_BYTES,
	PROPERTY_FORMAT_DATA16, 				// EVERYTHING3_PROPERTY_ID_FIRST_16_BYTES,
	PROPERTY_FORMAT_DATA32, 				// EVERYTHING3_PROPERTY_ID_FIRST_32_BYTES,
	PROPERTY_FORMAT_DATA64, 				// EVERYTHING3_PROPERTY_ID_FIRST_64_BYTES,
	PROPERTY_FORMAT_DATA128, 				// EVERYTHING3_PROPERTY_ID_FIRST_128_BYTES,
	PROPERTY_FORMAT_DATA1, 					// EVERYTHING3_PROPERTY_ID_LAST_BYTE,
	PROPERTY_FORMAT_DATA2, 					// EVERYTHING3_PROPERTY_ID_LAST_2_BYTES,
	PROPERTY_FORMAT_DATA4, 					// EVERYTHING3_PROPERTY_ID_LAST_4_BYTES,
	PROPERTY_FORMAT_DATA8, 					// EVERYTHING3_PROPERTY_ID_LAST_8_BYTES,
	PROPERTY_FORMAT_DATA16, 				// EVERYTHING3_PROPERTY_ID_LAST_16_BYTES,
	PROPERTY_FORMAT_DATA32, 				// EVERYTHING3_PROPERTY_ID_LAST_32_BYTES,
	PROPERTY_FORMAT_DATA64, 				// EVERYTHING3_PROPERTY_ID_LAST_64_BYTES,
	PROPERTY_FORMAT_DATA128, 				// EVERYTHING3_PROPERTY_ID_LAST_128_BYTES,
	PROPERTY_FORMAT_FORMATTED_TEXT12,			// EVERYTHING3_PROPERTY_ID_BYTE_ORDER_MARK,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_VOLUME_LABEL,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_FILE_LIST_PATH_AND_NAME,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_DISPLAY_PATH_AND_NAME,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PARSE_NAME,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_PARSE_PATH_AND_NAME,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_STEM,
	PROPERTY_FORMAT_HEX_NUMBER8,			// EVERYTHING3_PROPERTY_ID_SHELL_ATTRIBUTES,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_IS_FOLDER,
	PROPERTY_FORMAT_YESNO,					// EVERYTHING3_PROPERTY_ID_VALID_UTF8,
	PROPERTY_FORMAT_NOGROUPING_NUMBER3, 	// EVERYTHING3_PROPERTY_ID_STEM_LENGTH,
	PROPERTY_FORMAT_NOGROUPING_NUMBER1, 	// EVERYTHING3_PROPERTY_ID_EXTENSION_LENGTH,
	PROPERTY_FORMAT_NOGROUPING_NUMBER3, 	// EVERYTHING3_PROPERTY_ID_PATH_PART_LENGTH,
	PROPERTY_FORMAT_TIME, 					// EVERYTHING3_PROPERTY_ID_DATE_MODIFIED_TIME,
	PROPERTY_FORMAT_TIME, 					// EVERYTHING3_PROPERTY_ID_DATE_CREATED_TIME,
	PROPERTY_FORMAT_DATE, 					// EVERYTHING3_PROPERTY_ID_DATE_MODIFIED_DATE,
	PROPERTY_FORMAT_DATE, 					// EVERYTHING3_PROPERTY_ID_DATE_CREATED_DATE,
	PROPERTY_FORMAT_DATE, 					// EVERYTHING3_PROPERTY_ID_DATE_ACCESSED_DATE,
	PROPERTY_FORMAT_TIME, 					// EVERYTHING3_PROPERTY_ID_DATE_ACCESSED_TIME,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_PARENT_NAME,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_REPARSE_TARGET
	PROPERTY_FORMAT_GROUPING_NUMBER7,		// EVERYTHING3_PROPERTY_ID_DESCENDANT_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER7,		// EVERYTHING3_PROPERTY_ID_DESCENDANT_FOLDER_COUNT,
	PROPERTY_FORMAT_GROUPING_NUMBER7,		// EVERYTHING3_PROPERTY_ID_DESCENDANT_FILE_COUNT,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_FROM,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_TO,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_RECEIVED,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DATE_SENT,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_CONTAINER_FILENAMES,
	PROPERTY_FORMAT_GROUPING_NUMBER7,		// EVERYTHING3_PROPERTY_ID_CONTAINER_FILE_COUNT,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_0,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_1,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_2,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_3,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_4,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_5,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_6,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_7,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_8,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_9,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_ALLOCATION_SIZE,
	PROPERTY_FORMAT_DATA4, 					// EVERYTHING3_PROPERTY_ID_SFV_CRC32,
	PROPERTY_FORMAT_DATA16, 				// EVERYTHING3_PROPERTY_ID_MD5SUM_MD5,
	PROPERTY_FORMAT_DATA20, 				// EVERYTHING3_PROPERTY_ID_SHA1SUM_SHA1,
	PROPERTY_FORMAT_DATA32, 				// EVERYTHING3_PROPERTY_ID_SHA256SUM_SHA256,
	PROPERTY_FORMAT_YESNO, 					// EVERYTHING3_PROPERTY_ID_SFV_PASS,
	PROPERTY_FORMAT_YESNO, 					// EVERYTHING3_PROPERTY_ID_MD5SUM_PASS,
	PROPERTY_FORMAT_YESNO, 					// EVERYTHING3_PROPERTY_ID_SHA1SUM_PASS,
	PROPERTY_FORMAT_YESNO, 					// EVERYTHING3_PROPERTY_ID_SHA256SUM_PASS,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_ANSI,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_UTF8,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_UTF16LE,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_UTF16BE,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_TEXT_PLAIN,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_ALTERNATE_DATA_STREAM_HEX,
	PROPERTY_FORMAT_TEXT8,					// EVERYTHING3_PROPERTY_ID_PERCEIVED_TYPE,
	PROPERTY_FORMAT_TEXT24,					// EVERYTHING3_PROPERTY_ID_CONTENT_TYPE,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_OPENED_BY,
	PROPERTY_FORMAT_FORMATTED_TEXT12,			// EVERYTHING3_PROPERTY_ID_TARGET_MACHINE,
	PROPERTY_FORMAT_DATA64, 				// EVERYTHING3_PROPERTY_ID_SHA512SUM_SHA512,
	PROPERTY_FORMAT_YESNO, 					// EVERYTHING3_PROPERTY_ID_SHA512SUM_PASS,
	PROPERTY_FORMAT_TEXT64, 				// EVERYTHING3_PROPERTY_ID_PARENT_PATH,
	PROPERTY_FORMAT_DATA256, 				// EVERYTHING3_PROPERTY_ID_FIRST_256_BYTES,
	PROPERTY_FORMAT_DATA512, 				// EVERYTHING3_PROPERTY_ID_FIRST_512_BYTES,
	PROPERTY_FORMAT_DATA256, 				// EVERYTHING3_PROPERTY_ID_LAST_256_BYTES,
	PROPERTY_FORMAT_DATA512, 				// EVERYTHING3_PROPERTY_ID_LAST_512_BYTES,
	PROPERTY_FORMAT_YESNO, 					// EVERYTHING3_PROPERTY_ID_INDEX_ONLINE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_0,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_1,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_2,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_3,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_4,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_5,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_6,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_7,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_8,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_9,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_A,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_B,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_C,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_D,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_E,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_COLUMN_F,
	PROPERTY_FORMAT_FORMATTED_TEXT8,			// EVERYTHING3_PROPERTY_ID_ZONE_ID,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_REFERRER_URL,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_HOST_URL,
	PROPERTY_FORMAT_FORMATTED_TEXT12,			// EVERYTHING3_PROPERTY_ID_CHARACTER_ENCODING,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_ROOT_NAME,
	PROPERTY_FORMAT_VOLUME_SIZE,			// EVERYTHING3_PROPERTY_ID_USED_DISK_SIZE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_VOLUME_PATH,
	PROPERTY_FORMAT_NOGROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_MAX_CHILD_DEPTH,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_TOTAL_CHILD_SIZE,
	PROPERTY_FORMAT_NOGROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_ROW,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_CHILD_OCCURRENCE_COUNT,
	PROPERTY_FORMAT_TEXT48,					// EVERYTHING3_PROPERTY_ID_VOLUME_NAME,
	PROPERTY_FORMAT_GROUPING_NUMBER5,		// EVERYTHING3_PROPERTY_ID_DESCENDANT_OCCURRENCE_COUNT,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_OBJECT_ID,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_BIRTH_VOLUME_ID,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_BIRTH_OBJECT_ID,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_DOMAIN_ID,
	PROPERTY_FORMAT_DATA4,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_CRC32,
	PROPERTY_FORMAT_DATA8,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_CRC64,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_MD5,
	PROPERTY_FORMAT_DATA20,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_SHA1,
	PROPERTY_FORMAT_DATA32,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_SHA256,
	PROPERTY_FORMAT_DATA64,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_SHA512,
	PROPERTY_FORMAT_DATA4,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_CRC32,
	PROPERTY_FORMAT_DATA8,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_CRC64,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_MD5,
	PROPERTY_FORMAT_DATA20,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_SHA1,
	PROPERTY_FORMAT_DATA32,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_SHA256,
	PROPERTY_FORMAT_DATA64,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_SHA512,
	PROPERTY_FORMAT_DATA4,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_CRC32,
	PROPERTY_FORMAT_DATA8,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_CRC64,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_MD5,
	PROPERTY_FORMAT_DATA20,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_SHA1,
	PROPERTY_FORMAT_DATA32,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_SHA256,
	PROPERTY_FORMAT_DATA64,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_SHA512,
	PROPERTY_FORMAT_DATA4,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_CRC32_FROM_DISK,
	PROPERTY_FORMAT_DATA8,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_CRC64_FROM_DISK,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_MD5_FROM_DISK,
	PROPERTY_FORMAT_DATA20,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_SHA1_FROM_DISK,
	PROPERTY_FORMAT_DATA32,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_SHA256_FROM_DISK,
	PROPERTY_FORMAT_DATA64,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_SHA512_FROM_DISK,
	PROPERTY_FORMAT_DATA4,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_CRC32_FROM_DISK,
	PROPERTY_FORMAT_DATA8,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_CRC64_FROM_DISK,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_MD5_FROM_DISK,
	PROPERTY_FORMAT_DATA20,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_SHA1_FROM_DISK,
	PROPERTY_FORMAT_DATA32,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_SHA256_FROM_DISK,
	PROPERTY_FORMAT_DATA64,					// EVERYTHING3_PROPERTY_ID_FOLDER_DATA_AND_NAMES_SHA512_FROM_DISK,
	PROPERTY_FORMAT_DATA4,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_CRC32_FROM_DISK,
	PROPERTY_FORMAT_DATA8,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_CRC64_FROM_DISK,
	PROPERTY_FORMAT_DATA16,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_MD5_FROM_DISK,
	PROPERTY_FORMAT_DATA20,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_SHA1_FROM_DISK,
	PROPERTY_FORMAT_DATA32,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_SHA256_FROM_DISK,
	PROPERTY_FORMAT_DATA64,					// EVERYTHING3_PROPERTY_ID_FOLDER_NAMES_SHA512_FROM_DISK,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_LONG_NAME,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_LONG_PATH_AND_NAME,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_DIGITAL_SIGNATURE_NAME,
	PROPERTY_FORMAT_FILETIME,				// EVERYTHING3_PROPERTY_ID_DIGITAL_SIGNATURE_TIMESTAMP,
	PROPERTY_FORMAT_NOGROUPING_NUMBER1,		// EVERYTHING3_PROPERTY_ID_AUDIO_TRACK_COUNT,
	PROPERTY_FORMAT_NOGROUPING_NUMBER1,		// EVERYTHING3_PROPERTY_ID_VIDEO_TRACK_COUNT,
	PROPERTY_FORMAT_NOGROUPING_NUMBER2,		// EVERYTHING3_PROPERTY_ID_SUBTITLE_TRACK_COUNT,
	PROPERTY_FORMAT_TEXT16, 				// EVERYTHING3_PROPERTY_ID_NETWORK_INDEX_HOST,
	PROPERTY_FORMAT_TEXT64, 				// EVERYTHING3_PROPERTY_ID_ORIGINAL_LOCATION,
	PROPERTY_FORMAT_FILETIME, 				// EVERYTHING3_PROPERTY_ID_DATE_DELETED,
	PROPERTY_FORMAT_FORMATTED_TEXT32, 			// EVERYTHING3_PROPERTY_ID_STATUS,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_VORBIS_COMMENT,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_QUICKTIME_METADATA,
	PROPERTY_FORMAT_SIZE,					// EVERYTHING3_PROPERTY_ID_PARENT_SIZE,
	PROPERTY_FORMAT_VOLUME_SIZE,			// EVERYTHING3_PROPERTY_ID_ROOT_SIZE,
	PROPERTY_FORMAT_TEXT32,					// EVERYTHING3_PROPERTY_ID_OPENS_WITH,
	PROPERTY_FORMAT_HEX_NUMBER16,			// EVERYTHING3_PROPERTY_ID_RANDOMIZE,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_ICON,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_THUMBNAIL,
	PROPERTY_FORMAT_TEXT64,					// EVERYTHING3_PROPERTY_ID_CONTENT,
	PROPERTY_FORMAT_NONE,					// EVERYTHING3_PROPERTY_ID_SEPARATOR,
};

// default column widths.
static WORD _property_format_to_column_width[PROPERTY_FORMAT_COUNT] =
{
	0, // PROPERTY_FORMAT_NONE,
	8, // PROPERTY_FORMAT_TEXT8, // 8 characters
	10, // PROPERTY_FORMAT_TEXT10, // 10 characters
	12, // PROPERTY_FORMAT_TEXT12, // 8.3 characters
	16, // PROPERTY_FORMAT_TEXT16, // 16 characters
	24, // PROPERTY_FORMAT_TEXT24, // 24 characters
	30, // PROPERTY_FORMAT_TEXT30, // 30 characters
	32, // PROPERTY_FORMAT_TEXT32, // 32 characters
	47, // PROPERTY_FORMAT_TEXT47, // 47 characters
	48, // PROPERTY_FORMAT_TEXT48, // 48 characters
	64, // PROPERTY_FORMAT_TEXT64, // 64 characters
	14, // PROPERTY_FORMAT_SIZE, // 99,999,999,999
	19, // PROPERTY_FORMAT_VOLUME_SIZE, // 999,999,999,999,999
	4, // PROPERTY_FORMAT_EXTENSION, // 4 characters
	19, // PROPERTY_FORMAT_FILETIME, // 2025-06-16 21:22:23
	8, // PROPERTY_FORMAT_TIME, // 21:22:23
	8, // PROPERTY_FORMAT_DATE, // 2025-06-16
	4, // PROPERTY_FORMAT_ATTRIBUTES, // RASH
	1, // PROPERTY_FORMAT_NOGROUPING_NUMBER1, // 1
	2, // PROPERTY_FORMAT_NOGROUPING_NUMBER2, // 12
	3, // PROPERTY_FORMAT_NOGROUPING_NUMBER3, // 123
	4, // PROPERTY_FORMAT_NOGROUPING_NUMBER4, // 1234 (no comma)
	5, // PROPERTY_FORMAT_NOGROUPING_NUMBER5, // 12345 (no comma)
	2, // PROPERTY_FORMAT_GROUPING_NUMBER2, // 12
	3, // PROPERTY_FORMAT_GROUPING_NUMBER3, // 123
	5, // PROPERTY_FORMAT_GROUPING_NUMBER4, // 1,234
	6, // PROPERTY_FORMAT_GROUPING_NUMBER5, // 12,345
	7, // PROPERTY_FORMAT_GROUPING_NUMBER6, // 123,456
	9, // PROPERTY_FORMAT_GROUPING_NUMBER7, // 1,234,567
	8, // PROPERTY_FORMAT_KHZ, // 44.1 khz
	10, // PROPERTY_FORMAT_KBPS, // 99999 kbps
	5, // PROPERTY_FORMAT_RATING, // *****
	10, // PROPERTY_FORMAT_HEX_NUMBER8, // 0xdeadbeef
	18, // PROPERTY_FORMAT_HEX_NUMBER16, // 0xdeadbeefdeadbeef
	18, // PROPERTY_FORMAT_HEX_NUMBER32, // 0xdeadbeefdeadbeefdeadbeefdeadbeef. (ReFS fileid, NTFS still uses number16)
	9, // PROPERTY_FORMAT_DIMENSIONS, // 9999x9999
	6, // PROPERTY_FORMAT_F_STOP, // f/9.99
	9, // PROPERTY_FORMAT_EXPOSURE_TIME, // 1/350 sec
	8, // PROPERTY_FORMAT_ISO_SPEED, // ISO-9999
	11, // PROPERTY_FORMAT_EXPOSURE_BIAS, // +1.234 step
	10, // PROPERTY_FORMAT_FOCAL_LENGTH, // 999.999 mm
	9, // PROPERTY_FORMAT_SUBJECT_DISTANCE, // 999.999 m
	10, // PROPERTY_FORMAT_BCPS, // 9.999 bcps
	8, // PROPERTY_FORMAT_35MM_FOCAL_LENGTH, // 99999 mm
	14, // PROPERTY_FORMAT_ALTITUDE, // -9999.123456 m
	10, // PROPERTY_FORMAT_SEC, // 99.999 sec
	9, // PROPERTY_FORMAT_FIXED_Q1K, // -9999.123
	12, // PROPERTY_FORMAT_FIXED_Q1M, // -9999.123456
	8, // PROPERTY_FORMAT_DURATION, // 01:23:45
	2, // PROPERTY_FORMAT_DATA1, // data
	4, // PROPERTY_FORMAT_DATA2, // data
	8, // PROPERTY_FORMAT_DATA4, // crc
	16, // PROPERTY_FORMAT_DATA8, // crc64
	32, // PROPERTY_FORMAT_DATA16, // md5
	40, // PROPERTY_FORMAT_DATA20, // sha1
	64, // PROPERTY_FORMAT_DATA32, // sha256
	96, // PROPERTY_FORMAT_DATA48, // sha384
	128, // PROPERTY_FORMAT_DATA64, // sha512
	256, // PROPERTY_FORMAT_DATA128, // data
	512, // PROPERTY_FORMAT_DATA256, // data
	1024, // PROPERTY_FORMAT_DATA512, // data
	8, // PROPERTY_FORMAT_FORMATTED_TEXT8, // formatted 8 characters
	12, // PROPERTY_FORMAT_FORMATTED_TEXT12, // formatted 12 characters
	16, // PROPERTY_FORMAT_FORMATTED_TEXT16, // formatted 16 characters
	24, // PROPERTY_FORMAT_FORMATTED_TEXT24, // formatted 24 characters
	32, // PROPERTY_FORMAT_FORMATTED_TEXT32, // formatted 32 characters
	3, // PROPERTY_FORMAT_YESNO, // Yes/No
	4, // PROPERTY_FORMAT_PERCENT, // 100%
	5, // PROPERTY_FORMAT_ASPECT_RATIO, // 16:9 / 9.999
};

// right aligned columns
static BYTE _property_format_to_right_align[PROPERTY_FORMAT_COUNT] =
{
	0, // PROPERTY_FORMAT_NONE,
	0, // PROPERTY_FORMAT_TEXT8, // 8 characters
	0, // PROPERTY_FORMAT_TEXT10, // 10 characters
	0, // PROPERTY_FORMAT_TEXT12, // 8.3 characters
	0, // PROPERTY_FORMAT_TEXT16, // 16 characters
	0, // PROPERTY_FORMAT_TEXT24, // 24 characters
	0, // PROPERTY_FORMAT_TEXT30, // 30 characters
	0, // PROPERTY_FORMAT_TEXT32, // 32 characters
	0, // PROPERTY_FORMAT_TEXT47, // 47 characters
	0, // PROPERTY_FORMAT_TEXT48, // 48 characters
	0, // PROPERTY_FORMAT_TEXT64, // 64 characters
	1, // PROPERTY_FORMAT_SIZE, // 123456789
	1, // PROPERTY_FORMAT_VOLUME_SIZE, // 999,999,999,999,999
	0, // PROPERTY_FORMAT_EXTENSION, // 4 characters
	0, // PROPERTY_FORMAT_FILETIME, // 2025-06-16 21:22:23
	0, // PROPERTY_FORMAT_TIME, // 21:22:23
	0, // PROPERTY_FORMAT_DATE, // 2025-06-16
	0, // PROPERTY_FORMAT_ATTRIBUTES, // RASH
	1, // PROPERTY_FORMAT_NOGROUPING_NUMBER1, // 1
	1, // PROPERTY_FORMAT_NOGROUPING_NUMBER2, // 12
	1, // PROPERTY_FORMAT_NOGROUPING_NUMBER3, // 123
	1, // PROPERTY_FORMAT_NOGROUPING_NUMBER4, // 1234 (no comma)
	1, // PROPERTY_FORMAT_NOGROUPING_NUMBER5, // 12345 (no comma)
	1, // PROPERTY_FORMAT_GROUPING_NUMBER2, // 12
	1, // PROPERTY_FORMAT_GROUPING_NUMBER3, // 123
	1, // PROPERTY_FORMAT_GROUPING_NUMBER4, // 1,234
	1, // PROPERTY_FORMAT_GROUPING_NUMBER5, // 12,345
	1, // PROPERTY_FORMAT_GROUPING_NUMBER6, // 123,456
	1, // PROPERTY_FORMAT_GROUPING_NUMBER7, // 1,234,567
	1, // PROPERTY_FORMAT_KHZ, // 44.1 Khz
	1, // PROPERTY_FORMAT_KBPS, // 99999 kbps
	0, // PROPERTY_FORMAT_RATING, // *****
	1, // PROPERTY_FORMAT_HEX_NUMBER8, // 0xdeadbeef
	1, // PROPERTY_FORMAT_HEX_NUMBER16, // 0xdeadbeefdeadbeef
	1, // PROPERTY_FORMAT_HEX_NUMBER32, // 0xdeadbeefdeadbeefdeadbeefdeadbeef
	1, // PROPERTY_FORMAT_DIMENSIONS, // 9999x9999
	1, // PROPERTY_FORMAT_F_STOP, // f/9.99
	1, // PROPERTY_FORMAT_EXPOSURE_TIME, // 1/350 sec
	1, // PROPERTY_FORMAT_ISO_SPEED, // ISO-9999
	1, // PROPERTY_FORMAT_EXPOSURE_BIAS, // +1.234 step
	1, // PROPERTY_FORMAT_FOCAL_LENGTH, // 999.999 mm
	1, // PROPERTY_FORMAT_SUBJECT_DISTANCE, // 999.999 m
	1, // PROPERTY_FORMAT_BCPS, // 9.999 bcps
	1, // PROPERTY_FORMAT_35MM_FOCAL_LENGTH, // 99999 mm
	1, // PROPERTY_FORMAT_ALTITUDE, // 9999.123456 m
	1, // PROPERTY_FORMAT_SEC, // 99.999 sec
	1, // PROPERTY_FORMAT_FIXED_Q1K, // 9999.123
	1, // PROPERTY_FORMAT_FIXED_Q1M, // 9999.123456
	1, // PROPERTY_FORMAT_DURATION, // 1:23:45
	0, // PROPERTY_FORMAT_DATA1, // data
	0, // PROPERTY_FORMAT_DATA2, // data
	0, // PROPERTY_FORMAT_DATA4, // crc
	0, // PROPERTY_FORMAT_DATA8, // crc64
	0, // PROPERTY_FORMAT_DATA16, // md5
	0, // PROPERTY_FORMAT_DATA20, // sha1
	0, // PROPERTY_FORMAT_DATA32, // sha256
	0, // PROPERTY_FORMAT_DATA48, // sha384
	0, // PROPERTY_FORMAT_DATA64, // sha512
	0, // PROPERTY_FORMAT_DATA128, // data
	0, // PROPERTY_FORMAT_DATA256, // data
	0, // PROPERTY_FORMAT_DATA512, // data
	0, // PROPERTY_FORMAT_FORMATTED_TEXT8, // formatted 8 characters
	0, // PROPERTY_FORMAT_FORMATTED_TEXT12, // formatted 12 characters
	0, // PROPERTY_FORMAT_FORMATTED_TEXT16, // formatted 16 characters
	0, // PROPERTY_FORMAT_FORMATTED_TEXT24, // formatted 24 characters
	0, // PROPERTY_FORMAT_FORMATTED_TEXT32, // formatted 32 characters
	0, // PROPERTY_FORMAT_YESNO, // Yes/No
	1, // PROPERTY_FORMAT_PERCENT, // 100%
	1, // PROPERTY_FORMAT_ASPECT_RATIO, // 16:9 / 9.999
};

// default column widths.
static BYTE _property_format_to_default_sort_ascending[PROPERTY_FORMAT_COUNT] =
{
	1, // PROPERTY_FORMAT_NONE,
	1, // PROPERTY_FORMAT_TEXT8, // 8 characters
	1, // PROPERTY_FORMAT_TEXT10, // 10 characters
	1, // PROPERTY_FORMAT_TEXT12, // 8.3 characters
	1, // PROPERTY_FORMAT_TEXT16, // 16 characters
	1, // PROPERTY_FORMAT_TEXT24, // 24 characters
	1, // PROPERTY_FORMAT_TEXT30, // 30 characters
	1, // PROPERTY_FORMAT_TEXT32, // 32 characters
	1, // PROPERTY_FORMAT_TEXT47, // 47 characters
	1, // PROPERTY_FORMAT_TEXT48, // 48 characters
	1, // PROPERTY_FORMAT_TEXT64, // 64 characters
	0, // PROPERTY_FORMAT_SIZE, // 99,999,999,999
	0, // PROPERTY_FORMAT_VOLUME_SIZE, // 999,999,999,999,999
	1, // PROPERTY_FORMAT_EXTENSION, // 4 characters
	0, // PROPERTY_FORMAT_FILETIME, // 2025-06-16 21:22:23
	0, // PROPERTY_FORMAT_TIME, // 21:22:23
	0, // PROPERTY_FORMAT_DATE, // 2025-06-16
	1, // PROPERTY_FORMAT_ATTRIBUTES, // RASH
	0, // PROPERTY_FORMAT_NOGROUPING_NUMBER1, // 1
	0, // PROPERTY_FORMAT_NOGROUPING_NUMBER2, // 12
	0, // PROPERTY_FORMAT_NOGROUPING_NUMBER3, // 123
	0, // PROPERTY_FORMAT_NOGROUPING_NUMBER4, // 1234 (no comma)
	0, // PROPERTY_FORMAT_NOGROUPING_NUMBER5, // 12345 (no comma)
	0, // PROPERTY_FORMAT_GROUPING_NUMBER2, // 12
	0, // PROPERTY_FORMAT_GROUPING_NUMBER3, // 123
	0, // PROPERTY_FORMAT_GROUPING_NUMBER4, // 1,234
	0, // PROPERTY_FORMAT_GROUPING_NUMBER5, // 12,345
	0, // PROPERTY_FORMAT_GROUPING_NUMBER6, // 123,456
	0, // PROPERTY_FORMAT_GROUPING_NUMBER7, // 1,234,567
	0, // PROPERTY_FORMAT_KHZ, // 44.1 Khz
	0, // PROPERTY_FORMAT_KBPS, // 99999 kbps
	0, // PROPERTY_FORMAT_RATING, // *****
	0, // PROPERTY_FORMAT_HEX_NUMBER8, // 0xdeadbeef
	0, // PROPERTY_FORMAT_HEX_NUMBER16, // 0xdeadbeefdeadbeef
	0, // PROPERTY_FORMAT_HEX_NUMBER32, // 0xdeadbeefdeadbeefdeadbeefdeadbeef
	0, // PROPERTY_FORMAT_DIMENSIONS, // 9999x9999
	0, // PROPERTY_FORMAT_F_STOP, // f/9.99
	0, // PROPERTY_FORMAT_EXPOSURE_TIME, // 1/350 sec
	0, // PROPERTY_FORMAT_ISO_SPEED, // ISO-9999
	0, // PROPERTY_FORMAT_EXPOSURE_BIAS, // +1.234 step
	0, // PROPERTY_FORMAT_FOCAL_LENGTH, // 999.999 mm
	0, // PROPERTY_FORMAT_SUBJECT_DISTANCE, // 999.999 m
	0, // PROPERTY_FORMAT_BCPS, // 9.999 bcps
	0, // PROPERTY_FORMAT_35MM_FOCAL_LENGTH, // 99999 mm
	0, // PROPERTY_FORMAT_ALTITUDE, // 9999.123456 m
	0, // PROPERTY_FORMAT_SEC, // 99.999 sec
	0, // PROPERTY_FORMAT_FIXED_Q1K, // 9999.123
	0, // PROPERTY_FORMAT_FIXED_Q1M, // 9999.123456
	0, // PROPERTY_FORMAT_DURATION, // 1:23:45
	1, // PROPERTY_FORMAT_DATA1, // data
	1, // PROPERTY_FORMAT_DATA2, // data
	1, // PROPERTY_FORMAT_DATA4, // crc
	1, // PROPERTY_FORMAT_DATA8, // crc64
	1, // PROPERTY_FORMAT_DATA16, // md5
	1, // PROPERTY_FORMAT_DATA20, // sha1
	1, // PROPERTY_FORMAT_DATA32, // sha256
	1, // PROPERTY_FORMAT_DATA48, // sha384
	1, // PROPERTY_FORMAT_DATA64, // sha512
	1, // PROPERTY_FORMAT_DATA128, // data
	1, // PROPERTY_FORMAT_DATA256, // data
	1, // PROPERTY_FORMAT_DATA512, // data
	1, // PROPERTY_FORMAT_FORMATTED_TEXT8, // formatted 8 characters
	1, // PROPERTY_FORMAT_FORMATTED_TEXT12, // formatted 12 characters
	1, // PROPERTY_FORMAT_FORMATTED_TEXT16, // formatted 16 characters
	1, // PROPERTY_FORMAT_FORMATTED_TEXT24, // formatted 24 characters
	1, // PROPERTY_FORMAT_FORMATTED_TEXT32, // formatted 32 characters
	1, // PROPERTY_FORMAT_YESNO, // Yes/No
	0, // PROPERTY_FORMAT_PERCENT, // 100%
	0, // PROPERTY_FORMAT_ASPECT_RATIO, // 16:9 / 9.999
};

// returns a PROPERTY_FORMAT_* type.
// Returns PROPERTY_FORMAT_NONE for unknown property types.
BYTE property_get_format(DWORD property_id)
{
	if (property_id < EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT)	
	{
		return es_property_format[property_id];
	}

	return PROPERTY_FORMAT_NONE;
}

// returns TRUE if the column should be right aligned.
// Otherwise, returns FALSE.
BOOL property_is_right_aligned(DWORD property_id)
{
	if (property_id < EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT)
	{
		if (_property_format_to_right_align[property_get_format(property_id)])
		{
			return TRUE;
		}
	}
	else
	{
		const property_unknown_t *property_unknown;
		
		property_unknown = property_unknown_find(property_id);
		if (property_unknown)
		{
			if (property_unknown->is_right_aligned)
			{
				return TRUE;
			}
		}
	}
	
	return FALSE;
}

// get the property canonical name.
// stores the canonical name in out_wcbuf.
void property_get_canonical_name(DWORD property_id,wchar_buf_t *out_wcbuf)
{
	DEBUG_ASSERT((sizeof(_property_name_array) / sizeof(ES_UTF8*)) == EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT);

	if (property_id < EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT)
	{
		wchar_buf_copy_utf8_string(out_wcbuf,_property_name_array[property_id]);
	}
	else
	{
		const property_unknown_t *property_unknown;
		
		property_unknown = property_unknown_find(property_id);
		if (property_unknown)
		{
			wchar_buf_copy_utf8_string(out_wcbuf,PROPERTY_UNKNOWN_CANONICAL_NAME(property_unknown));
		}
		else
		{
			wchar_buf_empty(out_wcbuf);
		}
	}
}

// get the property canonical name.
// stores the canonical name in out_wcbuf.
void property_get_localized_name(DWORD property_id,wchar_buf_t *out_wcbuf)
{
	DEBUG_ASSERT((sizeof(_property_name_array) / sizeof(ES_UTF8*)) == EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT);

	// try localized name..
	// localizing EVERYTHING3_PROPERTY_ID_PATH_AND_NAME as "Path and Name" is horrible.
	// don't localize for now. 
	
	switch(property_id)
	{
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_0:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_1:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_2:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_3:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_4:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_5:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_6:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_7:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_8:
		case EVERYTHING3_PROPERTY_ID_CUSTOM_PROPERTY_9:

			{
				int got_localized_name;
				utf8_buf_t localized_name_cbuf;
				
				got_localized_name = 0;
				utf8_buf_init(&localized_name_cbuf);
				
				if (ipc3_get_property_localized_name(property_id,&localized_name_cbuf))
				{
					wchar_buf_copy_utf8_string_n(out_wcbuf,localized_name_cbuf.buf,localized_name_cbuf.length_in_bytes);
					
					got_localized_name = 1;
				}

				utf8_buf_kill(&localized_name_cbuf);
				
				if (got_localized_name)
				{
					return;
				}
			}

			break;
	}

	// try canonical name..

	if (property_id < EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT)
	{
		wchar_buf_copy_utf8_string(out_wcbuf,_property_name_array[property_id]);
	}
	else
	{
		const property_unknown_t *property_unknown;
		
		property_unknown = property_unknown_find(property_id);
		if (property_unknown)
		{
			wchar_buf_copy_utf8_string(out_wcbuf,PROPERTY_UNKNOWN_CANONICAL_NAME(property_unknown));
		}
		else
		{
			wchar_buf_empty(out_wcbuf);
		}
	}
}

// find a property ID an old column ID.
// returns EVERYTHING3_INVALID_PROPERTY_ID if the column ID is unknown.
DWORD property_id_from_old_column_id(int i)
{
	if (i < PROPERTY_OLD_COLUMN_COUNT)
	{
		return _property_old_column_id_to_property_id_array[i];
	}
	
	return EVERYTHING3_INVALID_PROPERTY_ID;
}

// compare a property name.
// ignores all dashes (-) in name.
// ignores '-', '.', '_' and ' ' in the search.
static int _property_name_compare(const ES_UTF8 *name,const wchar_t *search)
{
	const ES_UTF8 *name_p;
	const wchar_t *search_p;
	int bad_dash;
	
	name_p = name;
	search_p = search;
	bad_dash = 0;
	
	while(*search_p)
	{
		int c2;
		
		UTF8_STRING_GET_CHAR(search_p,c2);
		
		c2 = unicode_ascii_to_lower(c2);
		
		if ((c2 == '-') || (c2 == '.') || (c2 == '_') || (c2 == ' '))
		{
			if (*name_p == '-')
			{
				name_p++;
				continue;
			}
			
			bad_dash = 1;

			continue;
		}
		
		for(;;)
		{
			if (!*name_p)
			{
				// name < search
				return -1;
			}
			
			if (*name_p == '-')
			{
				// skip it.
			}
			else
			{
				break;
			}
			
			name_p++;
		}

		if (*name_p != c2)
		{
			return *name_p - c2;
		}
		
		name_p++;
	}
	
	if (*name_p)
	{
		// name > search.
		return 1;
	}
	
	if (bad_dash)
	{
		// name < search.
		return -1;
	}
	
	// match;
	return 0;
}

// binary search for a property by name
// returns the property ID if found.
// returns EVERYTHING3_INVALID_PROPERTY_ID if not found.
DWORD property_find(const wchar_t *s,int allow_property_system)
{
	if (*s)
	{
		SIZE_T blo;
		SIZE_T bhi;

		// make sure we are sorted by name (with no dashes '-')
	#ifdef _DEBUG
		
		{
			int i;
			utf8_buf_t last_cbuf;
			utf8_buf_t nodash_cbuf;
			
			utf8_buf_init(&last_cbuf);
			utf8_buf_init(&nodash_cbuf);
			
			for(i=0;i<PROPERTY_NAME_TO_ID_COUNT;i++)
			{
				utf8_buf_copy_utf8_string(&nodash_cbuf,_property_name_to_id_array[i].name);

				{
					ES_UTF8 *d;
					ES_UTF8 *p;
					d = nodash_cbuf.buf;
					p = nodash_cbuf.buf;
					while(*p)
					{
						if (*p == '-')
						{
						}
						else
						{
							*d++ = *p;
						}
						
						p++;
					}
					*d = 0;
					nodash_cbuf.length_in_bytes = d - nodash_cbuf.buf;
				}
				
				if (last_cbuf.length_in_bytes)
				{
					if (utf8_string_compare(last_cbuf.buf,nodash_cbuf.buf) > 0)
					{
						DEBUG_FATAL("property name is not sorted %s > %s\n",last_cbuf.buf,nodash_cbuf.buf);
					}
				}
				
				utf8_buf_copy_utf8_string_n(&last_cbuf,nodash_cbuf.buf,nodash_cbuf.length_in_bytes);
			}
			
			utf8_buf_kill(&nodash_cbuf);
			utf8_buf_kill(&last_cbuf);
		}

	#endif
		
		// search for name insertion point.
		blo = 0;
		bhi = PROPERTY_NAME_TO_ID_COUNT;

		while(blo < bhi)
		{
			SIZE_T bpos;
			int i;
			
			bpos = blo + ((bhi - blo) / 2);

			i = _property_name_compare(_property_name_to_id_array[bpos].name,s);
			if (i > 0)
			{
				bhi = bpos;
			}
			else
			if (!i)
			{
				return _property_name_to_id_array[bpos].id;
			}
			else
			{
				blo = bpos + 1;
			}
		}
	}

	if (allow_property_system)
	{
		DWORD ipc3_property_id;
		
		ipc3_property_id = ipc3_find_property(s);
		
		if (ipc3_property_id != EVERYTHING3_INVALID_PROPERTY_ID)
		{
			property_cache_unknown_information(ipc3_property_id);
			
			return ipc3_property_id;
		}
	}
	
	return EVERYTHING3_INVALID_PROPERTY_ID;
}

// get the default sort order.
// returns TRUE if ascending.
// Otherwise returns FALSE for descending order.
BOOL property_get_default_sort_ascending(DWORD property_id)
{
	if (property_id < EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT)
	{
		if (_property_format_to_default_sort_ascending[property_get_format(property_id)])
		{
			return TRUE;
		}
	}
	else
	{
		const property_unknown_t *property_unknown;
		
		property_unknown = property_unknown_find(property_id);
		if (property_unknown)
		{
			if (!property_unknown->is_sort_descending)
			{
				return TRUE;
			}
		}
	}
	
	return FALSE;
}

// returns the default column width for the specified property.
int property_get_default_width(DWORD property_id)
{
	if (property_id < EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT)
	{
		return _property_format_to_column_width[property_get_format(property_id)];
	}
	else
	{
		const property_unknown_t *property_unknown;
		
		property_unknown = property_unknown_find(property_id);
		if (property_unknown)
		{
			if (es_pixels_to_characters_div)
			{
				return (property_unknown->default_width * es_pixels_to_characters_mul) / es_pixels_to_characters_div;
			}
		}
	}
	
	return _property_format_to_column_width[PROPERTY_FORMAT_TEXT30];
}

void property_cache_unknown_information(DWORD property_id)
{
	if (property_id >= EVERYTHING3_PROPERTY_ID_BUILTIN_COUNT)
	{
		property_unknown_get(property_id);
	}
}				

void property_load_read_journal_names(void)
{
	// replace property names with the journal properties.
	_property_name_array[EVERYTHING3_PROPERTY_ID_PARENT_FILE_ID] = "Journal ID";
	_property_name_array[EVERYTHING3_PROPERTY_ID_FILE_ID] = "Change ID";
	_property_name_array[EVERYTHING3_PROPERTY_ID_TYPE] = "Action";
	_property_name_array[EVERYTHING3_PROPERTY_ID_DATE_INDEXED] = "Source Date Changed";
	_property_name_array[EVERYTHING3_PROPERTY_ID_DATE_RECENTLY_CHANGED] = "Parent Date Modified";
	_property_name_array[EVERYTHING3_PROPERTY_ID_PATH_AND_NAME] = "Filename";
	_property_name_array[EVERYTHING3_PROPERTY_ID_DATE_RUN] = "New Parent Date Modified";
	_property_name_array[EVERYTHING3_PROPERTY_ID_FILE_LIST_PATH_AND_NAME] = "New Filename";
}

